package gov.va.med.mhv.vitals.web.controller;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;

import javax.faces.application.FacesMessage;
import javax.faces.bean.ManagedBean;
import javax.faces.context.FacesContext;
import javax.faces.event.ComponentSystemEvent;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jfree.data.time.Day;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.xy.XYDataset;
import org.primefaces.component.datatable.DataTable;
import org.primefaces.event.data.SortEvent;
import org.primefaces.model.StreamedContent;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.vitals.dto.BodyTemperatureDTO;
import gov.va.med.mhv.vitals.enums.PeriodEnumeration;
import gov.va.med.mhv.vitals.util.CommonUtility;
import gov.va.med.mhv.vitals.web.util.WebUtility;

@ManagedBean
@Component
@Scope("session")
public class BodyTemperatureController extends AbstractController {

	private static final long serialVersionUID = -6192092835274409991L;
	private static Logger log = LogManager.getLogger(BodyTemperatureController.class);
	private List<BodyTemperatureDTO> tempreadings = new ArrayList<BodyTemperatureDTO>();
	private BodyTemperatureDTO selectedTempReading = new BodyTemperatureDTO();
	private BodyTemperatureDTO newTempReading = new BodyTemperatureDTO();
	private static final String TITLE = "Body Temperature";
	private static final String XAXIS = "Date";
	private static final String YAXIS = "Degrees(F)";

	public void init(ComponentSystemEvent event) {
		findUser();
		userprofileId = getUserProfileId();
		DataTable tempTable = (DataTable) FacesContext.getCurrentInstance().getViewRoot().findComponent("bodyTemperatureForm:btempList");
		if (!FacesContext.getCurrentInstance().isPostback()) {
			resetMessages();
			if (userprofileId != null) {
				tempreadings = findTempReadings(userprofileId);
			}
			setRowsPerPage(10);
			setPeriod(PeriodEnumeration.ONE_MONTH.getDescription());
		}
		else{
			if(sortColumn != null && sortBy != null){
				tempTable.setValueExpression("sortBy", sortColumn);
				tempTable.setSortOrder(sortBy);
			} 
		}
	}
	
	public void onSort(SortEvent event){
		sortColumn=event.getSortColumn().getValueExpression("sortBy");
		sortBy=event.isAscending()?"ascending":"descending";
	}

	private List<BodyTemperatureDTO> findTempReadings(Long userprofileId) {
		List<BodyTemperatureDTO> dtoList = null;
		try {

			dtoList = this.vitalSignsService.getTemperatureReadingsForUser(userprofileId);

		} catch (Exception e) {
			log.error("Error in Find Temperature readings:", e);
			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}

		return dtoList;
	}

	public String showDetail(BodyTemperatureDTO bodyTemperatureDTO) {
		resetMessages();
		selectedTempReading = bodyTemperatureDTO;
		setHourMinute(selectedTempReading);
		return "bodyTemperature";
	}
	
	public String editDisplay(BodyTemperatureDTO bodyTemperatureDTO) {
		resetMessages();
		selectedTempReading = bodyTemperatureDTO;
		setHourMinute(selectedTempReading);
		return "editBTDisplay";
	}
	
	public String resetEditDisplay(){
		return "editBTDisplay";
	}

	public String deleteDisplay(BodyTemperatureDTO bodyTemperatureDTO) {
		resetMessages();
		selectedTempReading = bodyTemperatureDTO;
		setHourMinute(selectedTempReading);
		setDeleteOrigin("tableView");
		return "deleteBTDisplay";
	}

	public String deleteRecordDisplay() {
		resetMessages();
		setDeleteOrigin(null);
		return "deleteBTDisplay";
	}

	private void setHourMinute(BodyTemperatureDTO selectedTempReading) {
		String dateTime = WebUtility.dateToString(selectedTempReading.getReading(), HOUR_MINUTE);
		if (!dateTime.equals(DEFAULT_DATETIME)) {
			selectedTempReading.setHour(WebUtility.dateToString(selectedTempReading.getReading(), HOURS));
			selectedTempReading.setMinute(WebUtility.dateToString(selectedTempReading.getReading(), MINUTES));
		}

	}

	public String addDisplay() {
		resetMessages();
		newTempReading = new BodyTemperatureDTO();
		newTempReading.setReading(new Date());
		return "addBTDisplay";
	}

	public String dashboardAddDisplay() {
		resetMessages();
		findUser();
		setRowsPerPage(10);
		userprofileId = getUserProfileId();
		if (userprofileId != null) {
			tempreadings = findTempReadings(userprofileId);
		}
		newTempReading = new BodyTemperatureDTO();
		newTempReading.setReading(new Date());
		FacesContext context = FacesContext.getCurrentInstance();
		context.getApplication().getNavigationHandler().handleNavigation(context, null,
				"/views/bodytemperature/addBTDisplay.xhtml");
		return null;
	}
	
	public String dashboardViewMore() {
		resetMessages();
		findUser();
		setRowsPerPage(10);
		userprofileId = getUserProfileId();
		if (userprofileId != null) {
			tempreadings = findTempReadings(userprofileId);
		}
		return "/views/bodytemperature/bodyTemperatureList";
	}

	public String showDahBoardDetail() {
		resetMessages();
		findUser();
		String btid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("btid");
		Long userProfileId = getUserProfileId();
		if (btid != null && userProfileId != null) {
			Long id = Long.valueOf(btid);
			selectedTempReading = getBTReadingById(userProfileId, id);
		}
		FacesContext context = FacesContext.getCurrentInstance();
		context.getApplication().getNavigationHandler().handleNavigation(context, null,
				"/views/bodytemperature/bodyTemperature.xhtml");
		return null;
	}

	private BodyTemperatureDTO getBTReadingById(Long userProfileId, Long id) {
		BodyTemperatureDTO dto = null;
		try {

			dto = this.vitalSignsService.getTemperatureReadingByid(userProfileId, id);

		} catch (Exception e) {
			log.error("Error in Find BT reading By id:", e);
			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}

		return dto;
	}

	public String delete() {
		String outcome = null;
		String btid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("btid");
		Long userProfileId = getUserProfileId();

		try {
			if (btid != null && userProfileId != null) {
				Long id = Long.valueOf(btid);

				this.vitalSignsService.deleteBodyTemperatureReading(userProfileId, id);
				tempreadings = findTempReadings(userprofileId);
				deleteMessage = true;
				outcome = "bodyTemperatureList";

			}
		} catch (Exception e) {
			log.error("Failed to Delete Tempertaure record", e);

			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}

		return outcome;
	}

	public String save() {
		String outcome = null;
		try {
			if (saveBTReading().getBodyTemperatureId() != null) {
				outcome = "bodyTemperatureList";
				tempreadings = findTempReadings(userprofileId);
				saveMessage = true;
			}
		} catch (MHVException e) {
			log.error("Failed to Save Tempertaure record", e);

			if (e.getValidationErrorMessages() != null) {
				addValidationMessages(e.getValidationErrorMessages());
			} else {
				FacesContext.getCurrentInstance().addMessage(null,
						new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
			}
		}
		return outcome;
	}

	public BodyTemperatureDTO saveBTReading() throws MHVException {
		resetMessages();
		String btid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("btid");
		Long id = null;
		BodyTemperatureDTO btreading = null;
		if (btid != null) {
			id = Long.valueOf(btid);
			btreading = getSelectedTempReading();
		} else {
			btreading = getNewTempReading();
		}

		prepareBTReading(id, btreading);
		Long bodyTempId =this.vitalSignsService.saveBodyTemperatureReading(btreading);
		btreading.setBodyTemperatureId(bodyTempId);
		return btreading;
		
	}

	private BodyTemperatureDTO prepareBTReading(Long id, BodyTemperatureDTO btreading) {
		Date dateTime = null;
		if (btreading.getReading() != null) {
			if (btreading.getHour() == null && btreading.getMinute() == null) {
				dateTime = WebUtility.getDaTeTime(btreading.getReading(), DEFAULT_HOUR, DEFAULT_MINUTE);
			} else {
				dateTime = WebUtility.getDaTeTime(btreading.getReading(), btreading.getHour(), btreading.getMinute());
			}
		}
		btreading.setReading(dateTime);
		btreading.setUserprofileId(getUserProfileId());
		return btreading;
	}

	public String saveAndAdd() {
		String outcome = null;
		try {
			if (saveBTReading().getBodyTemperatureId() != null) {
				outcome = "addBTDisplay";
				saveAndAddMessage = true;
				newTempReading = new BodyTemperatureDTO();
				newTempReading.setReading(new Date());
				tempreadings = findTempReadings(userprofileId);
			}

		} catch (MHVException e) {
			log.error("Failed to Save Tempertaure record", e);

			if (e.getValidationErrorMessages() != null) {
				addValidationMessages(e.getValidationErrorMessages());
			} else {
				FacesContext.getCurrentInstance().addMessage(null,
						new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
			}
		}
		return outcome;
	}

	public String updateLineModel() {
		return "bodyTemperatureGraph";
	}

	public StreamedContent getJchart() {
		StreamedContent content = null;
		try {
			content = createChart(TITLE, XAXIS, YAXIS);
		} catch (Exception e) {
			log.error("Failed to create Chart", e);
		}
		return content;
	}

	protected XYDataset createDataset(Set<String> uniqueYears) {
		TimeSeriesCollection dataset = new TimeSeriesCollection();
		TimeSeries series1 = new TimeSeries(YAXIS);
		String period = getPeriod();
		Date startDate = null;
		List<BodyTemperatureDTO> graphList;
		if (!period.equals(PeriodEnumeration.ALL_DATA.getDescription())) {
			startDate = getStartGraphDate(period);
		}
		if (startDate != null) {
			graphList = getGraphlist(tempreadings, startDate);
		} else {
			graphList = tempreadings;
		}
		for (BodyTemperatureDTO btreading : graphList) {

			series1.addOrUpdate(new Day(btreading.getReading()), btreading.getBodyTemperature());
			uniqueYears
					.add(CommonUtility.dateToString(btreading.getReading(), YYYY_MM_DD_HHMMSS_FORMAT).substring(0, 4));
		}
		dataset.addSeries(series1);
		return dataset;
	}

	private List<BodyTemperatureDTO> getGraphlist(List<BodyTemperatureDTO> btreadings, Date startDate) {
		List<BodyTemperatureDTO> graphList = new ArrayList<BodyTemperatureDTO>();
		Integer startdateValue = Integer.valueOf(CommonUtility.dateToString(startDate, YYYYMMDD_FORMAT));
		for (BodyTemperatureDTO btreading : btreadings) {
			Integer readingValue = Integer.valueOf(CommonUtility.dateToString(btreading.getReading(), YYYYMMDD_FORMAT));
			if (readingValue >= startdateValue) {
				graphList.add(btreading);
			}
		}
		return graphList;
	}

	public String printerFriendlySummary(){
		resetMessages();
		return "printBTSummary";
	}

	public List<BodyTemperatureDTO> getTempreadings() {
		return tempreadings;
	}

	public void setTempreadings(List<BodyTemperatureDTO> tempreadings) {
		this.tempreadings = tempreadings;
	}

	public BodyTemperatureDTO getSelectedTempReading() {
		return selectedTempReading;
	}

	public void setSelectedTempReading(BodyTemperatureDTO selectedTempReading) {
		this.selectedTempReading = selectedTempReading;
	}

	public BodyTemperatureDTO getNewTempReading() {
		return newTempReading;
	}

	public void setNewTempReading(BodyTemperatureDTO newTempReading) {
		this.newTempReading = newTempReading;
	}

}
